; Making use of NASM's power
;
; For a long time, assembly language programmers did everything the hard way.
; Indeed, it was only recently really that the use of high-level constructs
; like switch, for and while were introduced into assembly lanugage. In the
; past, assembly programmers have been tempted to do everything the hard
; way and shun these techniques for fear of losing a cycle where one could
; have been saved.
;
; Times are changing though and assembly language must change with it.
;
; Today's programmers don't expect, and nor should they, to have to write
; everything themselves. Compilers have advanced a great deal in the way
; things can be automated but assembly language has lagged, perhaps because
; of its traditional hardcore appearance.
;
; MASM was one of the first to make the bold move in this direction with it's
; conditional .IF statements and powerful macro handling, and Win32asm
; programmers have been quick to capitalise on these abilities as they make
; coding in assembler much nicer. There's no point in forcing the programmer
; to write all the mind-numbing switch or loop logic when these things are
; done the same way every time. If you think about it, how much time do you
; waste writing the assembler code for yet another loop or if-then clause?
;
; NASM has none of these features, but NASM has a very substantial advantage
; over MASM - it has an extremely powerful macro processor. So powerful in
; fact that it can be used to emulate most things other assemblers can do
; anyway. This makes it much more extensible and better - if we don't like
; the way something's implemented we can always change it.
;
; When writing Windows applications, case type statements crop up on a
; regular basis. For this (and the following) example I'll show the typical
; HLL code in C style though the syntax is usually very similar for any HLL:
;
; switch (i) {
;     case 0:
;         // i = 0
;         break;
;     case 1:
;         // i = 1
;         break;
;     default:
;         // i != 0 && i != 1
; }
;
; In assembler, we _might_ implement this like:
;

_switch:
	mov	eax, i			; eax holds variable
	cmp	eax, 0			; case 0?
	jne	_switch_1		; no, check next case
	; case 0 code goes here
	jmp	_switch_end		; go to end of switch
_switch_1:
	cmp	eax, 1			; case 1?
	jne	_switch_default		; no, onto the default case
	; case 1 code goes here
	jmp	_switch_end		; go to end of switch
_switch_default:
	; default case code goes here
_switch_end:

;
; Note the _might_ - a compiler could well implement this as a jump table
; instead, though such a simple example as the one above might not be any
; more efficient, indeed it would probably be slower (assuming the jump
; table hasn't been cached, a plausible scenario).
;
; It only takes a quick glance to see how much work we have to actually do
; to generate something so simple and in Windows applications these are
; very common. However, with appropriate macros we can reduce this to the
; following instead:
;

	switch eax
		case 0
			; case 0 code goes here
			break
		case 1
			; case 1 code goes here
			break
		default
			; default case code goes here
	switchend

;
; The full syntax is:
;

	switch <switch variable>
		case <value 1>, <value 2>, ..., <value n>
			;
			; code to be executed if switch variable equals any of the specified values
			;
			break
			;
			; break forces the switch statement to end. If you omit
			; this processing will continue at the next case or range statement
			;
		caseand <value 1>, <value 2>, ..., <value n>
			;
			; code to be executed if switch variable equals all the specified values,
			; of use if you want to check several registers or memory locations
			; all equal the switch variable. This and rangeand (below) are included
			; more for completeness and because in rare situations you might want them
			;
			break
		range <lower bound of range 1>, <upper bound of range 1>, ..., <lower bound of range n>, <upper bound of range n>
			;
			; code to be executed if switch variable lies between any of
			; the ranges specified (including the bounds themselves)
			;
			break
		rangeand <lower bound of range 1>, <upper bound of range 1>, ..., <lower bound of range n>, <upper bound of range n>
			;
			; code to be executed if switch variable lies between all the ranges.
			; like caseand only of use rarely, and included for completeness.
			;
			break
		default
			;
			; code to be executed if no case value matches
			;
	switchend

;
; Note that even though I use the term 'value', this doesn't mean only literal
; values like 1, 2, etc. but should be taken to mean any valid assembler
; expression, eg a register or memory variable. Likewise, I use the term
; variable in many places which should also be interpreted as being able
; to take any valid assembler expression. If in doubt, try it! You can only
; get assembler errors after all...
;
; The actual code produced is (though the labels would in fact be
; uniquely generated by the assembler):
;

_case0:
	cmp	eax, 0
	jne	_case1
	; case 0 code goes here
	jmp	_case_end
_case1:
	cmp	eax, 1
	jne	_case2
	; case 1 code goes here
	jmp	_case_end
_case2:
	; default code goes here
_case_end:

;
; I think the benefits of this are clear:
;
;     o Code is far more readable and intuitive
;
;     o Code is much easier to modify to add new cases or move cases
;       around
;
;     o No need to code the (same each time) logic for the statements
;
;     o Less room for making logic errors (which can be notoriously
;       difficult to track down) in the main code
;
;     o No need to hunt for new (descriptive!) variable names
;
;     o Almost the same as C, which much Win32 code is written in
;
; A companion of the switch block is the if/else combination, which
; would typically be used like (in C style):
;
; if (i == 5) {
;     // i == 5
; } else {
;     // i != 5
; }
;
; In assembler we can do something similar:
;

	if eax, e, ebx
		; eax == ebx
	else
		; eax != ebx
	endif

;
; The full syntax is:
;

	if <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>
		;
		; code to be executed if any of the comparisons are true
		;
	else
		;
		; code to be executed if all of the comparisons are false
		;
	endif

	ifand <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>
		;
		; code to be executed if all of the comparisons are true
		;
	else
	endif
;
; The three parameters for the if macro are <value 1 to compare>, <comparison
; operator> and <value 2 to compare>. The comparison operator is the condition
; code representing the comparison you'd like to make, ie instead of je you'd
; pass e. The code generated would be:
;

	cmp	eax, ebx
	jne	_if1
	; eax == ebx
	jmp	_if2
_if1:
	; eax != ebx
_if2:

;
; Note how the j?? after the comparison is the _inverse_ of the condition code
; passed to the routine. This ensures that the code is executed in an
; intuitive way (ie if the comparison using the comparison operator we pass is
; true then don't branch). This sounds way more complex than it is, I'm sure
; you can get the idea.
;
; A close relative of these branching statements is the for loop construct,
; which looks like (in C):
;
; for(i=0;i<j;i++) {
;     // code to be executed each iteration
; }
;
; Which we make as:
;

	for eax, 0, le, ebx, 1
		; code to be executed each iteration
	next

;
; The above may look slightly confusing, but this should make it clear:
;

	for <loop variable>, <initial variable count>, <comparison operator>, <value to compare loop variable to>, <step>
		;
		; code to be executed each loop
		;
	next

;
; And now the last of the looping/branching constructs - the while/wend
; loop and its counterpart the repeat/until loop:
;
; while (i < 5) {
;     // code to be executed while i < 5
; }
;
; repeat {
;     // code to be executed until i == 10
; } until (i == 10)
;
; In assembler, these become:
;

	while eax, l, 5
		; code to repeat while eax is less than 5
	wend

	repeat
		; code to repeat until eax equals 10
	until eax, e, 10

;
; The full syntax being:
;

	while <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>
		;
		; code to be executed if any of the comparisons are true
		;
	wend

	whileand <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>
		;
		; code to be executed if all of the comparisons are true
		;
	wend

	repeat
		;
		; code to be executed until any of the comparisons are true
		;
	until <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>

	repeat
		;
		; code to be executed until all of the comparisons are true
		;
	untiland <value 1a>, <comparison operator>, <value 1b>, <value 2a>, <comparison operator>, <value 2b>, ..., <value na>, <comparison operator>, <value nb>

;
; Like the if construct, both while and until can take multiple comparisons
; and work just the same way.
;
; Now, one of the most important constructs in your arsenal - the
; procedure construct:
;
; void myFun(int var1, int var2, int var3) {
;     int i,j,k,l;
; }
;
; Which in assembler becomes:
;

	proc myFun, var1, var, var3
		ddlocal i,j,k,l
		endlocals
	endproc

;
; The full syntax is:
;

	proc <proc name>, <parameter 1 name>, <parameter 2 name>, ..., <parameter n name>
		;
		; There is an alternate version of proc called procglobal that is identical to
		; proc in every respect except that it also generates a label of the form
		; _<proc name>@<parameter count * 4> so, procglobal _test, var1, var2 would
		; generate __test@8, and also makes the label global. This is useful when
		; writing any function that must be exported, like DLL functions or
		; program entry points
		;
		; dblocal, dwlocal, ddlocal and dqlocal reserve space for local variables of
		; byte, word, dword and quadword size respectively
		;
		dblocal <local byte name 1>, <local byte name 2>, ..., <local byte name n>
		dwlocal <local word name 1>, <local word name 2>, ..., <local word name n>
		ddlocal <local dword name 1>, <local dword name 2>, ..., <local dword name n>
		dqlocal <local qword name 1>, <local qword name 2>, ..., <local qword name n>
		;
		; dbstatic, dwstatic, ddstatic and dqstatic reserve space for local variables of
		; byte, word, dword and quadword size respectively that retain their values
		; between procedure calls.
		;
		dbstatic <static byte name 1>, <static byte name 2>, ..., <static byte name n>
		dwstatic <static word name 1>, <static word name 2>, ..., <static word name n>
		ddstatic <static dword name 1>, <static dword name 2>, ..., <static dword name n>
		dqstatic <static qword name 1>, <static qword name 2>, ..., <static qword name n>
		;
		; dbglobal, dwglobal, ddglobal and dqglobal reserve space for global variables of
		; byte, word, dword and quadword size respectively that can be referenced from
		; anywhere in the application. Note that these can be declared outside of
		; procedures, but I document them within for ease instead.
		;
		dbglobal <global byte name 1>, <global byte name 2>, ..., <global byte name n>
		dwglobal <global word name 1>, <global word name 2>, ..., <global word name n>
		ddglobal <global dword name 1>, <global dword name 2>, ..., <global dword name n>
		dqglobal <global qword name 1>, <global qword name 2>, ..., <global qword name n>
		;
		; struclocal reserves space for local structures with the names specified
		;
		struclocal <local structure name 1>, <structure>, ..., <local structure name n>, <structure>
		;
		; strucstatic reserve space for local structures that retain their valus
		; between procedure calls
		;
		strucstatic <static structure name 1>, <structure>, ..., <static structure name n>, <structure>
		;
		; strucglobal reserves space for global structures with the names specified
		; that can be referenced from anywhere in the application. Like dnglobal these
		; can be declared outside procedures
		;
		strucglobal <global structure name 1>, <structure>, ..., <global structure name n>, <structure>
		;
		; bufflocal reserves space for local buffers with the names and sizes specified
		;
		bufflocal <local buffer name 1>, <buffer size (bytes)>, ... <local buffer name n>, <buffer size (bytes)>
		;
		; buffstatic reserves space for local buffers that retain their values between
		; procedure calls
		;
		buffstatic <static buffer name 1>, <buffer size (bytes)>, ... <static buffer name n>, <buffer size (bytes)>
		;
		; buffglobal reserves space for global buffers with the names and sizes specified
		; that can be referenced from anywhere in the application. Like dnglobal these
		; can be declared outside procedures
		;
		buffglobal <global buffer name 1>, <buffer size (bytes)>, ... <global buffer name n>, <buffer size (bytes)>
		;
		; endlocals _must_ be called after all local variables have been declared.
		; it reserves the actual stack space, zeroes it all out and sets it up
		; for easy access to structures
		;
		endlocals
	endproc

;
; The following rules apply when referring to variables, buffers, etc:
;
;   Local variables:
;     Obtain the contents of the variable by prefixing the name with a dot character
;     Obtain the offset from ebp by prefixing the name with two dots
;
;   Static variables:
;     Obtain the contents of the variable by prefixing the name with a dot character
;     Obtain the address by prefixing the name with two dots
;
;   Global variables:
;     Obtain the contents of the variable by prefixing the name with a dot character
;     Obtain the address by prefixing the name with two dots
;       _or_ by just using the variable name itself
;
;   Local structures:
;     Obtain the address of the structure by prefixing the name with a dot character
;     Obtain the offset from ebp by prefixing the name with two dots
;
;   Global structures:
;     Obtain the address of the structure by prefixing the name with a dot character
;
;   Local buffers:
;     Obtain the address of the buffer by prefixing the name with a dot character
;     Obtain the offset from ebp by prefixing the name with two dots
;
;   Global buffers:
;     Obtain the address of the buffer by prefixing the name with a dot character
;
; A stupid example of some of these constructs would be:
;
	proc myFun, var1, var2, var3
		ddlocal i,j,k,l
		struclocal n,MYSTRUC
		bufflocal o,128
		ddstatic p,q
		endlocals
		mov	eax, .var1		; eax holds value of var1
		mov	eax, .var2		; eax holds value of var2
		mov	eax, .var3		; eax holds value of var3
		mov	esi, .n			; esi -> local structure MYSTRUC
		mov	edi, .o			; edi -> local buffer o (128 bytes in size)
		mov	.i, 0			; local variable i holds 0
		mov	.j, 0			; local variable j holds 0
		mov	.k, 0			; local variable k holds 0
		mov	.l, 0			; local variable l holds 0
		mov	.p, dword 0		; static variable p holds 0
		mov	.q, dword 0		; static variable q holds 0
	endproc

;
; There are several important things to note:
;
;   1. No need to specify the variable size as this was set when the
;      variable was declared. If you try and use a variable with the
;      wrong size you'll get a type mismatch error reported by NASM.
;
;   2. All local variables, structures and buffers are initialised to
;      zero. Statics and globals are not.
;
;   3. For alignment purposes, byte and word variables take up a dword
;      and buffer sizes are rounded up to the nearest dword boundary.
;
;   4. Passed parameters are assumed to be dword in size.
;
;   5. At the end of the procedure, parameters passed will be popped
;      from the stack, as per the Win32 stdcall convention (the called
;      function cleans up the stack, rather than the caller)
;
; It's very important that you understand how to use procedures, local
; and global variables - they're used extensively in the examples and
; provide a huge degree of flexibility when writing your own programs.
;
